package com.yd.common.utils;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 对象转换工具类
 * 提供将Java对象转换为Map的功能
 */
public class BeanMapUtils {

    /**
     * 将任意对象转换为Map<String, Object>
     * 包含对象的所有字段（包括私有字段）
     *
     * @param <T> 对象类型
     * @param obj 要转换的对象
     * @return 包含对象字段名和值的Map
     * @throws IllegalAccessException 如果无法访问字段
     */
    public static <T> Map<String, Object> convertToMap(T obj) throws IllegalAccessException {
        Map<String, Object> map = new HashMap<>();

        if (obj == null) {
            return map;
        }

        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();

        for (Field field : fields) {
            field.setAccessible(true); // 允许访问私有字段
            Object value = field.get(obj);
            map.put(field.getName(), value);
        }

        return map;
    }

    /**
     * 将任意对象转换为Map<String, Object>
     * 包含对象的所有字段，包括继承的字段
     *
     * @param <T> 对象类型
     * @param obj 要转换的对象
     * @return 包含对象字段名和值的Map
     * @throws IllegalAccessException 如果无法访问字段
     */
    public static <T> Map<String, Object> convertToMapIncludingInherited(T obj) throws IllegalAccessException {
        Map<String, Object> map = new HashMap<>();

        if (obj == null) {
            return map;
        }

        Class<?> clazz = obj.getClass();
        while (clazz != null && clazz != Object.class) {
            Field[] fields = clazz.getDeclaredFields();

            for (Field field : fields) {
                field.setAccessible(true);
                Object value = field.get(obj);
                map.put(field.getName(), value);
            }

            clazz = clazz.getSuperclass(); // 处理父类的字段
        }

        return map;
    }

    /**
     * 安全地将对象转换为Map，遇到异常时返回空Map
     *
     * @param <T> 对象类型
     * @param obj 要转换的对象
     * @return 包含对象字段名和值的Map，转换失败时返回空Map
     */
    public static <T> Map<String, Object> convertToMapSafely(T obj) {
        try {
            return convertToMap(obj);
        } catch (IllegalAccessException e) {
            // 记录日志或处理异常
            return new HashMap<>();
        }
    }

    /**
     * 将对象列表转换为Map列表
     *
     * @param <T> 对象类型
     * @param objList 要转换的对象列表
     * @return 包含对象字段名和值的Map列表
     * @throws IllegalAccessException 如果无法访问字段
     */
    public static <T> List<Map<String, Object>> convertListToMapList(List<T> objList) throws IllegalAccessException {
        List<Map<String, Object>> resultList = new ArrayList<>();

        if (objList == null) {
            return resultList;
        }

        for (T obj : objList) {
            Map<String, Object> map = convertToMap(obj);
            resultList.add(map);
        }

        return resultList;
    }

    /**
     * 将对象列表转换为Map列表（包含继承字段）
     *
     * @param <T> 对象类型
     * @param objList 要转换的对象列表
     * @return 包含对象字段名和值的Map列表
     * @throws IllegalAccessException 如果无法访问字段
     */
    public static <T> List<Map<String, Object>> convertListToMapListIncludingInherited(List<T> objList)
            throws IllegalAccessException {
        List<Map<String, Object>> resultList = new ArrayList<>();

        if (objList == null) {
            return resultList;
        }

        for (T obj : objList) {
            Map<String, Object> map = convertToMapIncludingInherited(obj);
            resultList.add(map);
        }

        return resultList;
    }

    /**
     * 安全地将对象列表转换为Map列表，遇到异常时跳过有问题的对象
     *
     * @param <T> 对象类型
     * @param objList 要转换的对象列表
     * @return 包含对象字段名和值的Map列表
     */
    public static <T> List<Map<String, Object>> convertListToMapListSafely(List<T> objList) {
        List<Map<String, Object>> resultList = new ArrayList<>();

        if (objList == null) {
            return resultList;
        }

        for (T obj : objList) {
            try {
                Map<String, Object> map = convertToMap(obj);
                resultList.add(map);
            } catch (IllegalAccessException e) {
                // 记录日志或处理异常，跳过有问题的对象
                // 可以选择添加空Map或跳过
            }
        }

        return resultList;
    }

    /**
     * 安全地将对象列表转换为Map列表，遇到异常时添加空Map
     *
     * @param <T> 对象类型
     * @param objList 要转换的对象列表
     * @return 包含对象字段名和值的Map列表
     */
    public static <T> List<Map<String, Object>> convertListToMapListSafelyWithEmpty(List<T> objList) {
        List<Map<String, Object>> resultList = new ArrayList<>();

        if (objList == null) {
            return resultList;
        }

        for (T obj : objList) {
            Map<String, Object> map = convertToMapSafely(obj);
            resultList.add(map);
        }

        return resultList;
    }
}
