package com.yd.feign.config;

import feign.RequestInterceptor;
import feign.RequestTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import javax.servlet.http.HttpServletRequest;

@Component
public class FeignTokenInterceptor implements RequestInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(FeignTokenInterceptor.class);

    // 使用ThreadLocal存储Token，用于异步场景
    private static final ThreadLocal<String> THREAD_LOCAL_TOKEN = new ThreadLocal<>();

    @Override
    public void apply(RequestTemplate template) {
        // 首先尝试从ThreadLocal获取（异步场景）
        String threadLocalToken = THREAD_LOCAL_TOKEN.get();
        if (threadLocalToken != null && !threadLocalToken.trim().isEmpty()) {
            template.header("Authorization", threadLocalToken);
            logger.debug("从ThreadLocal添加Authorization头到Feign请求");
            return;
        }

        // ThreadLocal没有，再从请求上下文获取（同步场景）
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();

        if (attributes == null) {
            logger.debug("RequestContextHolder没有找到请求属性，跳过Token传递");
            return;
        }

        HttpServletRequest request = attributes.getRequest();
        String token = request.getHeader("Authorization");

        if (token != null && !token.trim().isEmpty()) {
            template.header("Authorization", token);
            logger.debug("已添加Authorization头到Feign请求");
        } else {
            logger.debug("当前请求中没有Authorization头，跳过Token传递");
        }
    }

    // 设置ThreadLocal Token的方法
    public static void setThreadLocalToken(String token) {
        THREAD_LOCAL_TOKEN.set(token);
    }

    // 清理ThreadLocal Token的方法
    public static void clearThreadLocalToken() {
        THREAD_LOCAL_TOKEN.remove();
    }
}
