package com.yd.gateway.filter;

import com.yd.auth.core.security.DummyAuthenticationManager;
import com.yd.auth.core.utils.JwtUtils;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import io.jsonwebtoken.Claims;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.web.server.WebFilterExchange;
import org.springframework.security.web.server.authentication.AuthenticationWebFilter;
import org.springframework.security.web.server.authentication.ServerAuthenticationConverter;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;
import com.alibaba.fastjson.JSON;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 网关认证过滤器
 */
@Component
public class AuthenticationFilter extends AuthenticationWebFilter {

    private final JwtUtils jwtUtils;

    public AuthenticationFilter(JwtUtils jwtUtils) {
        super(new DummyAuthenticationManager());
        this.jwtUtils = jwtUtils;
        setServerAuthenticationConverter(jwtAuthenticationConverter());
    }

    /**
     * JWT认证转换器
     */
    private ServerAuthenticationConverter jwtAuthenticationConverter() {
        return exchange -> {
            ServerHttpRequest request = exchange.getRequest();
            String token = getTokenFromRequest(request);
            
            if (token == null) {
                return Mono.error(new BusinessException("未提供令牌"));
            }
            
            try {
                // 验证令牌
                if (!jwtUtils.validateToken(token)) {
                    return Mono.error(new BusinessException("无效的令牌"));
                }
                
                // 解析令牌
                Claims claims = jwtUtils.parseToken(token);
                String username = claims.getSubject();
                
                // 获取角色信息
                List<String> roles = (List<String>) claims.get("roles");
                List<SimpleGrantedAuthority> authorities = new ArrayList<>();
                
                if (roles != null && !roles.isEmpty()) {
                    authorities = roles.stream()
                            .map(role -> new SimpleGrantedAuthority("ROLE_" + role))
                            .collect(Collectors.toList());
                }
                
                // 设置认证信息
                UsernamePasswordAuthenticationToken authentication = 
                    new UsernamePasswordAuthenticationToken(username, null, authorities);
                
                return Mono.just(authentication);
            } catch (Exception e) {
                return Mono.error(new BusinessException(e.getMessage()));
            }
        };
    }

    /**
     * 从请求头中获取令牌
     */
    private String getTokenFromRequest(ServerHttpRequest request) {
        String bearerToken = request.getHeaders().getFirst(HttpHeaders.AUTHORIZATION);
        if (bearerToken != null && bearerToken.startsWith("Bearer ")) {
            return bearerToken.substring(7);
        }
        return null;
    }

    /**
     * 处理认证失败
     */
    protected Mono<Void> onAuthenticationFailure(WebFilterExchange webFilterExchange, AuthenticationException exception) {
        ServerWebExchange exchange = webFilterExchange.getExchange();
        exchange.getResponse().setStatusCode(HttpStatus.UNAUTHORIZED);
        exchange.getResponse().getHeaders().setContentType(MediaType.APPLICATION_JSON);
        
        Result<?> result = Result.fail(401, exception.getMessage());
        byte[] bytes = JSON.toJSONString(result).getBytes(StandardCharsets.UTF_8);
        DataBuffer buffer = exchange.getResponse().bufferFactory().wrap(bytes);
        
        return exchange.getResponse().writeWith(Mono.just(buffer));
    }
}
