package com.yd.auth.core.service.impl;

import com.yd.auth.core.dto.LoginRequest;
import com.yd.auth.core.dto.LoginResponse;
import com.yd.auth.core.security.JwtTokenProvider;
import com.yd.auth.core.service.AuthService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

@Service
public class AuthServiceImpl implements AuthService {

    private final AuthenticationManager authenticationManager;
    private final JwtTokenProvider jwtTokenProvider;

    @Autowired
    public AuthServiceImpl(AuthenticationManager authenticationManager, 
                          JwtTokenProvider jwtTokenProvider) {
        this.authenticationManager = authenticationManager;
        this.jwtTokenProvider = jwtTokenProvider;
    }

    @Override
    public LoginResponse login(LoginRequest loginRequest) {
        // 1. 创建认证对象
        Authentication authentication = new UsernamePasswordAuthenticationToken(
            loginRequest.getUsername(), 
            loginRequest.getPassword()
        );
        
        // 2. 进行认证
        Authentication authenticated = authenticationManager.authenticate(authentication);
        
        // 3. 设置安全上下文
        SecurityContextHolder.getContext().setAuthentication(authenticated);
        
        // 4. 生成令牌
        String token = jwtTokenProvider.generateToken(authenticated);
        
        // 5. 创建响应对象
        LoginResponse response = new LoginResponse();
        response.setToken(token);
        response.setExpiresIn(jwtTokenProvider.getJwtExpiration());
        
        return response;
    }

    @Override
    public void logout(String token) {
        // 实现令牌失效逻辑（可选）
        // 实际应用中可能需要将令牌加入黑名单
        SecurityContextHolder.clearContext();
    }

    @Override
    public Authentication getAuthentication(String token) {
        if (jwtTokenProvider.validateToken(token)) {
            return jwtTokenProvider.getAuthentication(token);
        }
        return null;
    }
}
