package com.yd.auth.core.security;

import com.yd.auth.core.utils.JwtUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

@Component
public class JwtAuthenticationProvider implements AuthenticationProvider {

    @Autowired
    private JwtUtils jwtUtils;

    @Autowired
    private UserDetailsService userDetailsService;

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        String token = (String) authentication.getCredentials();
        
        try {
            // 验证JWT令牌
            if (!jwtUtils.validateToken(token)) {
                throw new BadCredentialsException("无效的令牌");
            }
            
            // 从令牌中获取用户名
            String username = jwtUtils.getUsernameFromToken(token);
            
            // 加载用户详情
            UserDetails userDetails = userDetailsService.loadUserByUsername(username);
            
            // 从令牌中获取角色
            List<String> roles = jwtUtils.getRolesFromToken(token);
            List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();
            
            for (String role : roles) {
                authorities.add(new SimpleGrantedAuthority(role));
            }
            
            // 创建已认证的令牌
            JwtAuthenticationToken authenticatedToken = 
                new JwtAuthenticationToken(userDetails, authorities);
            authenticatedToken.setDetails(authentication.getDetails());
            
            return authenticatedToken;
        } catch (Exception e) {
            throw new BadCredentialsException("认证失败", e);
        }
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return JwtAuthenticationToken.class.isAssignableFrom(authentication);
    }
}
