package com.yd.common.utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 树形工具类 - 用于将扁平列表转换为树形结构
 */
public class TreeUtils {

    /**
     * 将扁平列表转换为树形结构
     * 
     * @param list 原始数据列表
     * @param idGetter 获取节点ID的方法
     * @param parentIdGetter 获取父节点ID的方法
     * @param childrenSetter 设置子节点的方法
     * @param rootParentId 根节点的父ID值（通常为0或null）
     * @param <T> 节点类型
     * @param <R> ID类型
     * @return 树形结构列表
     */
    public static <T, R> List<T> buildTree(
            List<T> list,
            Function<T, R> idGetter,
            Function<T, R> parentIdGetter,
            ChildrenSetter<T> childrenSetter,
            R rootParentId) {

        // 1. 创建节点映射表（ID -> 节点）
        Map<R, T> nodeMap = list.stream()
                .collect(Collectors.toMap(idGetter, Function.identity()));

        // 2. 创建结果列表（根节点列表）
        List<T> rootNodes = new ArrayList<>();

        // 3. 遍历所有节点，构建父子关系
        for (T node : list) {
            R parentId = parentIdGetter.apply(node);
            
            // 根节点处理
            if (Objects.equals(parentId, rootParentId)) {
                rootNodes.add(node);
                continue;
            }

            // 非根节点处理
            T parentNode = nodeMap.get(parentId);
            if (parentNode != null) {
                List<T> children = childrenSetter.getChildren(parentNode);
                if (children == null) {
                    children = new ArrayList<>();
                    childrenSetter.setChildren(parentNode, children);
                }
                children.add(node);
            }
        }

        // 4. 对树进行排序（可选）
        sortTree(rootNodes, childrenSetter);

        return rootNodes;
    }

    /**
     * 递归排序树形结构（按orderNum排序）
     * 
     * @param nodes 当前层节点列表
     * @param childrenSetter 子节点访问器
     * @param <T> 节点类型
     */
    public static <T> void sortTree(List<T> nodes, ChildrenSetter<T> childrenSetter) {
        if (nodes == null || nodes.isEmpty()) {
            return;
        }
        
        // 按orderNum排序（如果没有orderNum属性，需要自定义排序逻辑）
        nodes.sort((a, b) -> {
            try {
                // 使用反射获取orderNum属性
                java.lang.reflect.Method getOrderNum = a.getClass().getMethod("getOrderNum");
                Integer orderNumA = (Integer) getOrderNum.invoke(a);
                Integer orderNumB = (Integer) getOrderNum.invoke(b);
                return Integer.compare(orderNumA == null ? 0 : orderNumA, 
                                       orderNumB == null ? 0 : orderNumB);
            } catch (Exception e) {
                // 如果没有orderNum属性，则按ID排序
                try {
                    java.lang.reflect.Method getId = a.getClass().getMethod("getId");
                    Comparable idA = (Comparable) getId.invoke(a);
                    Comparable idB = (Comparable) getId.invoke(b);
                    return idA.compareTo(idB);
                } catch (Exception ex) {
                    return 0;
                }
            }
        });
        
        // 递归排序子节点
        for (T node : nodes) {
            List<T> children = childrenSetter.getChildren(node);
            if (children != null && !children.isEmpty()) {
                sortTree(children, childrenSetter);
            }
        }
    }

    /**
     * 从树形结构中查找指定节点
     * 
     * @param tree 树形结构
     * @param targetId 目标ID
     * @param idGetter ID获取器
     * @param childrenGetter 子节点获取器
     * @param <T> 节点类型
     * @param <R> ID类型
     * @return 找到的节点，未找到返回null
     */
    public static <T, R> T findNodeInTree(
            List<T> tree, 
            R targetId,
            Function<T, R> idGetter,
            Function<T, List<T>> childrenGetter) {
        
        for (T node : tree) {
            // 检查当前节点
            if (Objects.equals(idGetter.apply(node), targetId)) {
                return node;
            }
            
            // 递归检查子节点
            List<T> children = childrenGetter.apply(node);
            if (children != null && !children.isEmpty()) {
                T found = findNodeInTree(children, targetId, idGetter, childrenGetter);
                if (found != null) {
                    return found;
                }
            }
        }
        return null;
    }

    /**
     * 将树形结构扁平化为列表
     * 
     * @param tree 树形结构
     * @param childrenGetter 子节点获取器
     * @param <T> 节点类型
     * @return 扁平化列表
     */
    public static <T> List<T> flattenTree(
            List<T> tree, 
            Function<T, List<T>> childrenGetter) {
        
        List<T> result = new ArrayList<>();
        flattenTree(tree, childrenGetter, result);
        return result;
    }

    private static <T> void flattenTree(
            List<T> nodes, 
            Function<T, List<T>> childrenGetter, 
            List<T> result) {
        
        if (nodes == null) return;
        
        for (T node : nodes) {
            result.add(node);
            List<T> children = childrenGetter.apply(node);
            if (children != null && !children.isEmpty()) {
                flattenTree(children, childrenGetter, result);
            }
        }
    }

    /**
     * 子节点设置器接口
     * 
     * @param <T> 节点类型
     */
    @FunctionalInterface
    public interface ChildrenSetter<T> {
        /**
         * 设置子节点列表
         * 
         * @param parent 父节点
         * @param children 子节点列表
         */
        void setChildren(T parent, List<T> children);
        
        /**
         * 获取子节点列表
         * 
         * @param parent 父节点
         * @return 子节点列表
         */
        default List<T> getChildren(T parent) {
            try {
                // 使用反射获取children属性
                java.lang.reflect.Method getChildren = parent.getClass().getMethod("getChildren");
                return (List<T>) getChildren.invoke(parent);
            } catch (Exception e) {
                return null;
            }
        }
    }

    /**
     * 创建树形菜单的JSON结构（带层级路径）
     * 
     * @param tree 树形结构
     * @param childrenGetter 子节点获取器
     * @param mapper 节点转换器
     * @param <T> 原始节点类型
     * @param <R> 结果类型
     * @return 树形JSON结构
     */
    public static <T, R> List<Map<String, Object>> createTreeJson(
            List<T> tree, 
            Function<T, List<T>> childrenGetter,
            Function<T, R> mapper,
            String pathSeparator) {
        
        List<Map<String, Object>> result = new ArrayList<>();
        createTreeJson(tree, childrenGetter, mapper, result, "", pathSeparator);
        return result;
    }

    private static <T, R> void createTreeJson(
            List<T> nodes, 
            Function<T, List<T>> childrenGetter,
            Function<T, R> mapper,
            List<Map<String, Object>> result,
            String parentPath,
            String pathSeparator) {
        
        if (nodes == null) return;
        
        for (T node : nodes) {
            Map<String, Object> nodeMap = new HashMap<>();
            
            // 添加节点数据
            nodeMap.put("data", mapper.apply(node));
            
            // 创建当前节点路径
            try {
                java.lang.reflect.Method getId = node.getClass().getMethod("getId");
                Object id = getId.invoke(node);
                String currentPath = parentPath.isEmpty() ? id.toString() : 
                    parentPath + pathSeparator + id;
                nodeMap.put("path", currentPath);
            } catch (Exception e) {
                nodeMap.put("path", parentPath);
            }
            
            // 处理子节点
            List<T> children = childrenGetter.apply(node);
            if (children != null && !children.isEmpty()) {
                List<Map<String, Object>> childrenJson = new ArrayList<>();
                createTreeJson(children, childrenGetter, mapper, childrenJson, 
                             (String) nodeMap.get("path"), pathSeparator);
                nodeMap.put("children", childrenJson);
            }
            
            result.add(nodeMap);
        }
    }
}
