package com.yd.common.utils;

import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Objects;

public class DateUtil {

    public static String formatDate(Object date) {
        if (date == null) return "";
        if (date instanceof Date) {
            return new SimpleDateFormat("yyyy/MM/dd").format((Date) date);
        }
        return date.toString();
    }

    public static Date parseDate(String dateStr) {
        try {
            return new SimpleDateFormat("yyyy/MM/dd").parse(dateStr);
        } catch (ParseException e) {
            return new Date();
        }
    }

    /**
     * localDateTime转成年/月/日这种格式
     * @param localDateTime
     * @return
     */
    public static String getyyyyMMdd(LocalDateTime localDateTime) {
        if (Objects.isNull(localDateTime)) {
            return "";
        }

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd");
        return localDateTime.format(formatter);
    }

    /**
     * localDateTime转成时分这种格式
     * @param localDateTime
     * @return
     */
    public static String getHHmm(LocalDateTime localDateTime) {
        if (Objects.isNull(localDateTime)) {
            return "";
        }

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
        return localDateTime.format(formatter);
    }

    /**
     * localDateTime转成年-月-日 时:分这种格式
     * @param localDateTime
     * @return
     */
    public static String getyyyyMMddHHmm(LocalDateTime localDateTime) {
        if (Objects.isNull(localDateTime)) {
            return "";
        }

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        return localDateTime.format(formatter);
    }

    /**
     * yyyy/MM/dd HH:mm转成LocalDateTime
     * @param timeStr
     * @return
     */
    public static LocalDateTime getLocalDateTime(String timeStr) {
        if (StringUtils.isBlank(timeStr)) {
            return null;
        }

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm");
        // 解析字符串为LocalDateTime
        LocalDateTime localDateTime = LocalDateTime.parse(timeStr, formatter);
        return localDateTime;
    }

    /**
     * LocalDateTime转成Date
     * @return
     */
    public static Date convertDateByLocalDateTime(LocalDateTime localDateTime){
        //指定时区（如系统默认时区）
        ZoneId zoneId = ZoneId.systemDefault();
        //转换为 ZonedDateTime，再转为 Instant，最后生成 Date
        Date date = Date.from(localDateTime.atZone(zoneId).toInstant());
        return date;
    }
}
