package com.yd.base.feign.response.exchangerate;

import lombok.Data;
import java.math.BigDecimal;
import java.util.Date;

/**
 * 汇率转换响应DTO
 * @author zxm
 * @since 2025-12-29
 */
@Data
public class ApiExchangeRateConvertResponse {

    /**
     * 原始金额
     */
    private BigDecimal originalAmount;

    /**
     * 原始币种代码
     */
    private String originalCurrency;

    /**
     * 转换后金额
     */
    private BigDecimal convertedAmount;

    /**
     * 目标币种代码
     */
    private String targetCurrency;

    /**
     * 汇率（原始币种 -> 目标币种）
     */
    private BigDecimal exchangeRate;

    /**
     * 反向汇率（目标币种 -> 原始币种）
     */
    private BigDecimal reverseExchangeRate;

    /**
     * 汇率类型
     */
    private String rateType;

    /**
     * 生效日期
     */
    private Date effectiveDate;

    /**
     * 交易日期
     */
    private Date transactionDate;

    /**
     * 汇率来源
     */
    private String rateSource;

    /**
     * 转换时间
     */
    private Date convertTime;

    /**
     * 是否使用交叉汇率计算
     */
    private Boolean isCrossRate = false;

    /**
     * 交叉汇率中间币种
     */
    private String crossRateCurrency;

    /**
     * 转换状态
     */
    private String convertStatus;

    /**
     * 错误码
     */
    private String errorCode;

    /**
     * 错误信息
     */
    private String errorMessage;

    /**
     * 汇率记录ID
     */
    private Long exchangeRateId;

    /**
     * 基础币种名称
     */
    private String baseCurrencyName;

    /**
     * 目标币种名称
     */
    private String targetCurrencyName;

    /**
     * 业务场景标识
     */
    private String businessScene;

    /**
     * 业务ID
     */
    private String businessId;

    /**
     * 请求ID
     */
    private String requestId;

    /**
     * 扩展字段（JSON格式）
     */
    private String extInfo;

    /**
     * 转换详情
     */
    @Data
    public static class ConvertDetail {
        /**
         * 直接汇率是否存在
         */
        private Boolean directRateExists;

        /**
         * 是否使用反向汇率
         */
        private Boolean useReverseRate;

        /**
         * 汇率计算路径
         */
        private String ratePath;

        /**
         * 汇率更新时间
         */
        private Date rateUpdateTime;

        /**
         * 汇率有效期
         */
        private Date rateExpireTime;
    }

    /**
     * 转换详情（当needDetail=true时返回）
     */
    private ConvertDetail convertDetail;

    /**
     * 创建成功响应
     */
    public static ApiExchangeRateConvertResponse success(
            BigDecimal originalAmount,
            String originalCurrency,
            BigDecimal convertedAmount,
            String targetCurrency,
            BigDecimal exchangeRate) {

        ApiExchangeRateConvertResponse response = new ApiExchangeRateConvertResponse();
        response.setOriginalAmount(originalAmount);
        response.setOriginalCurrency(originalCurrency);
        response.setConvertedAmount(convertedAmount);
        response.setTargetCurrency(targetCurrency);
        response.setExchangeRate(exchangeRate);
        response.setConvertTime(new Date());
        response.setConvertStatus("SUCCESS");

        // 计算反向汇率
        if (exchangeRate != null && exchangeRate.compareTo(BigDecimal.ZERO) != 0) {
            try {
                BigDecimal reverseRate = BigDecimal.ONE.divide(
                        exchangeRate, 8, BigDecimal.ROUND_HALF_UP);
                response.setReverseExchangeRate(reverseRate);
            } catch (Exception e) {
                // 忽略计算错误
            }
        }

        return response;
    }

    /**
     * 创建失败响应
     */
    public static ApiExchangeRateConvertResponse error(String errorCode, String errorMessage) {
        ApiExchangeRateConvertResponse response = new ApiExchangeRateConvertResponse();
        response.setConvertTime(new Date());
        response.setConvertStatus("FAILED");
        response.setErrorCode(errorCode);
        response.setErrorMessage(errorMessage);
        return response;
    }
}