package com.yd.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.base.api.service.ApiSpeciesService;
import com.yd.base.feign.request.species.ApiSpeciesAddRequest;
import com.yd.base.feign.request.species.ApiSpeciesEditRequest;
import com.yd.base.feign.request.species.ApiSpeciesPageRequest;
import com.yd.base.feign.response.species.ApiSpeciesDetailResponse;
import com.yd.base.feign.response.species.ApiSpeciesPageResponse;
import com.yd.base.service.dto.SpeciesDto;
import com.yd.base.service.model.Species;
import com.yd.base.service.service.ISpeciesService;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class ApiSpeciesServiceImpl implements ApiSpeciesService {

    @Autowired
    private ISpeciesService iSpeciesService;

    /**
     * 分页列表查询-规格
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSpeciesPageResponse>> page(ApiSpeciesPageRequest request) {
        Page<ApiSpeciesPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiSpeciesPageResponse> iPage = iSpeciesService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-规格
     * @param request
     * @return
     */
    @Override
    public Result add(ApiSpeciesAddRequest request) {
        //校验规格值唯一性
        List<Species> speciesList =  iSpeciesService.queryList(SpeciesDto.builder()
                .value(request.getValue())
                .build());
        if (!CollectionUtils.isEmpty(speciesList)) {
            throw new BusinessException("规格值已存在");
        }
        Species species = new Species();
        BeanUtils.copyProperties(request,species);
        species.setSpeciesBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_SPECIES.getCode()));
        iSpeciesService.saveOrUpdate(species);
        return Result.success();
    }

    /**
     * 编辑-规格
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiSpeciesEditRequest request) {
        //校验规格是否存在
        Result<Species> result = checkSpeciesIsExist(request.getSpeciesBizId());
        Species species = result.getData();
        //校验规格值唯一性
        List<Species> speciesList =  iSpeciesService.queryList(SpeciesDto.builder()
                .value(request.getValue())
                .speciesBizId(request.getSpeciesBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(speciesList)) {
            throw new BusinessException("规格值已存在");
        }
        BeanUtils.copyProperties(request,species);
        iSpeciesService.saveOrUpdate(species);
        return Result.success();
    }

    /**
     * 详情-规格
     * @param speciesBizId 规格表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiSpeciesDetailResponse> detail(String speciesBizId) {
        //校验规格是否存在
        Result<Species> result = checkSpeciesIsExist(speciesBizId);
        Species species = result.getData();
        ApiSpeciesDetailResponse response = new ApiSpeciesDetailResponse();
        BeanUtils.copyProperties(species,response);
        return Result.success(response);
    }

    /**
     * 删除-规格 TODO
     * @param speciesBizId 规格表唯一业务ID
     * @return
     */
    @Override
    public Result del(String speciesBizId) {
        return null;
    }

    /**
     * 校验规格对象是否存在
     * @param speciesBizId
     * @return
     */
    public Result<Species> checkSpeciesIsExist(String speciesBizId) {
        Species species = iSpeciesService.queryOne(speciesBizId);
        if (Objects.isNull(species)) {
            //数据不存在
            throw new BusinessException("规格对象数据不存在");
        }
        return Result.success(species);
    }
}
