package com.yd.base.api.service.impl;

import com.yd.base.api.service.ApiExchangeRateService;
import com.yd.base.feign.request.exchangerate.ApiExchangeRateConvertRequest;
import com.yd.base.feign.response.exchangerate.ApiExchangeRateConvertResponse;
import com.yd.base.service.model.ExchangeRate;
import com.yd.base.service.service.IExchangeRateService;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;

@Slf4j
@Service
public class ApiExchangeRateServiceImpl implements ApiExchangeRateService {

    @Autowired
    private IExchangeRateService iExchangeRateService;

    private static final String DEFAULT_BASE_CURRENCY = "USD";
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final SimpleDateFormat SIMPLE_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    /**
     * 货币转换
     * @param request
     * @return
     */
    @Override
    public Result<ApiExchangeRateConvertResponse> convert(ApiExchangeRateConvertRequest request) {
        try {
            log.info("货币转换请求: {}", request);

            // 参数验证
            if (request.getAmount() == null) {
                return Result.fail("金额不能为空");
            }
            if (StringUtils.isBlank(request.getFromCurrency())) {
                return Result.fail("源币种不能为空");
            }
            if (StringUtils.isBlank(request.getToCurrency())) {
                return Result.fail("目标币种不能为空");
            }

            // 验证币种是否支持
            if (!isCurrencySupported(request.getFromCurrency())) {
                return Result.fail("源币种不支持: " + request.getFromCurrency());
            }
            if (!isCurrencySupported(request.getToCurrency())) {
                return Result.fail("目标币种不支持: " + request.getToCurrency());
            }

            // 获取转换日期
            LocalDate effectiveDate = getEffectiveDate(request.getTransactionDate(), request.getDate());

            // 执行转换
            BigDecimal convertedAmount = convertAmount(
                    request.getAmount(),
                    request.getFromCurrency(),
                    request.getToCurrency(),
                    effectiveDate
            );

            // 获取汇率
            BigDecimal exchangeRate = getExchangeRate(
                    request.getFromCurrency(),
                    request.getToCurrency(),
                    effectiveDate
            );

            // 构建响应
            ApiExchangeRateConvertResponse response = new ApiExchangeRateConvertResponse();
            response.setOriginalAmount(request.getAmount());
            response.setOriginalCurrency(request.getFromCurrency());
            response.setConvertedAmount(convertedAmount);
            response.setTargetCurrency(request.getToCurrency());
            response.setExchangeRate(exchangeRate);
            response.setTransactionDate(request.getTransactionDate());

            log.info("货币转换结果: {} {} = {} {}, 汇率: {}",
                    request.getAmount(), request.getFromCurrency(),
                    convertedAmount, request.getToCurrency(), exchangeRate);

            return Result.success(response);
        } catch (BusinessException e) {
            log.error("货币转换业务异常", e);
            return Result.fail(e.getMessage());
        } catch (Exception e) {
            log.error("货币转换系统异常", e);
            return Result.fail("货币转换失败: " + e.getMessage());
        }
    }

    /**
     * 获取汇率
     * @param baseCurrency 基础币种
     * @param targetCurrency 目标币种
     * @param date
     * @return
     */
    @Override
    public Result<BigDecimal> getExchangeRate(String baseCurrency, String targetCurrency, String date) {
        try {
            log.info("获取汇率: {} -> {}, date: {}", baseCurrency, targetCurrency, date);

            if (StringUtils.isBlank(baseCurrency)) {
                return Result.fail("基础币种不能为空");
            }
            if (StringUtils.isBlank(targetCurrency)) {
                return Result.fail("目标币种不能为空");
            }

            // 解析日期
            LocalDate effectiveDate = parseDate(date);

            BigDecimal rate = getExchangeRate(baseCurrency, targetCurrency, effectiveDate);

            if (rate == null) {
                return Result.fail("汇率不存在");
            }

            return Result.success(rate);
        } catch (BusinessException e) {
            log.error("获取汇率业务异常", e);
            return Result.fail(e.getMessage());
        } catch (Exception e) {
            log.error("获取汇率系统异常", e);
            return Result.fail("获取汇率失败: " + e.getMessage());
        }
    }

    /**
     * 批量转换
     * @param requests
     * @return
     */
    @Override
    public Result<List<ApiExchangeRateConvertResponse>> batchConvert(List<ApiExchangeRateConvertRequest> requests) {
        try {
            log.info("批量货币转换: {}条记录", requests == null ? 0 : requests.size());

            if (CollectionUtils.isEmpty(requests)) {
                return Result.success(new ArrayList<>());
            }

            List<ApiExchangeRateConvertResponse> responses = new ArrayList<>();

            for (ApiExchangeRateConvertRequest request : requests) {
                try {
                    // 参数验证
                    if (request.getAmount() == null ||
                            StringUtils.isBlank(request.getFromCurrency()) ||
                            StringUtils.isBlank(request.getToCurrency())) {
                        log.warn("批量转换中跳过无效请求: {}", request);
                        continue;
                    }

                    // 获取转换日期
                    LocalDate effectiveDate = getEffectiveDate(request.getTransactionDate(), request.getDate());

                    // 执行转换
                    BigDecimal convertedAmount = convertAmount(
                            request.getAmount(),
                            request.getFromCurrency(),
                            request.getToCurrency(),
                            effectiveDate
                    );

                    // 获取汇率
                    BigDecimal exchangeRate = getExchangeRate(
                            request.getFromCurrency(),
                            request.getToCurrency(),
                            effectiveDate
                    );

                    // 构建响应
                    ApiExchangeRateConvertResponse response = new ApiExchangeRateConvertResponse();
                    response.setOriginalAmount(request.getAmount());
                    response.setOriginalCurrency(request.getFromCurrency());
                    response.setConvertedAmount(convertedAmount);
                    response.setTargetCurrency(request.getToCurrency());
                    response.setExchangeRate(exchangeRate);
                    response.setTransactionDate(request.getTransactionDate());

                    responses.add(response);
                } catch (Exception e) {
                    log.error("批量转换中单条转换失败: {}", request, e);
                    // 跳过失败记录
                }
            }

            return Result.success(responses);
        } catch (Exception e) {
            log.error("批量货币转换失败", e);
            return Result.fail("批量货币转换失败: " + e.getMessage());
        }
    }

    /**
     * 验证币种是否支持
     * @param currency
     * @return
     */
    @Override
    public Result<Boolean> isCurrencySupportedResult(String currency) {
        try {
            log.info("验证币种是否支持: {}", currency);

            if (StringUtils.isBlank(currency)) {
                return Result.success(false);
            }

            boolean isSupported = isCurrencySupported(currency);

            return Result.success(isSupported);
        } catch (Exception e) {
            log.error("验证币种支持失败", e);
            return Result.fail("验证币种支持失败: " + e.getMessage());
        }
    }

    /**
     * 获取支持的币种列表
     * @return
     */
    @Override
    public Result<List<String>> getSupportedCurrencies() {
        try {
            log.info("获取支持的币种列表");

            List<String> currencies = getSupportedCurrenciesList();

            return Result.success(currencies);
        } catch (Exception e) {
            log.error("获取支持币种列表失败", e);
            return Result.fail("获取支持币种列表失败: " + e.getMessage());
        }
    }

    // ==================== 业务逻辑方法 ====================

    /**
     * 货币转换（业务逻辑）
     */
    private BigDecimal convertAmount(BigDecimal amount, String fromCurrency, String toCurrency, LocalDate effectiveDate) {
        if (amount == null) {
            return BigDecimal.ZERO;
        }

        if (StringUtils.isBlank(fromCurrency) || StringUtils.isBlank(toCurrency)) {
            throw new BusinessException("币种代码不能为空");
        }

        // 如果币种相同，直接返回
        if (fromCurrency.equalsIgnoreCase(toCurrency)) {
            return amount;
        }

        // 获取汇率
        BigDecimal exchangeRate = getExchangeRate(fromCurrency, toCurrency, effectiveDate);

        // 计算转换后的金额
        BigDecimal convertedAmount = amount.multiply(exchangeRate);

        // 设置精度（保留2位小数，银行家舍入法）
        return convertedAmount.setScale(2, RoundingMode.HALF_EVEN);
    }

    /**
     * 获取汇率（业务逻辑）
     */
    private BigDecimal getExchangeRate(String baseCurrency, String targetCurrency, LocalDate effectiveDate) {
        if (StringUtils.isBlank(baseCurrency) || StringUtils.isBlank(targetCurrency)) {
            throw new BusinessException("币种代码不能为空");
        }

        // 如果币种相同，返回1
        if (baseCurrency.equalsIgnoreCase(targetCurrency)) {
            return BigDecimal.ONE;
        }

        // 1. 尝试获取直接汇率
        BigDecimal directRate = getDirectExchangeRate(baseCurrency, targetCurrency, effectiveDate);
        if (directRate != null) {
            return directRate;
        }

        // 2. 尝试获取反向汇率
        BigDecimal reverseRate = getDirectExchangeRate(targetCurrency, baseCurrency, effectiveDate);
        if (reverseRate != null) {
            // 计算反向汇率的倒数
            return BigDecimal.ONE.divide(reverseRate, 8, RoundingMode.HALF_UP);
        }

        // 3. 尝试通过中间货币计算交叉汇率
        return calculateCrossRate(baseCurrency, targetCurrency, effectiveDate);
    }

    /**
     * 获取直接汇率
     */
    private BigDecimal getDirectExchangeRate(String baseCurrency, String targetCurrency, LocalDate effectiveDate) {
        ExchangeRate exchangeRate;

        if (effectiveDate != null) {
            // 查询指定日期的汇率
            exchangeRate = iExchangeRateService.getByCurrencyAndDate(baseCurrency, targetCurrency, effectiveDate);
        } else {
            // 查询最新汇率
            exchangeRate = iExchangeRateService.getLatestByCurrency(baseCurrency, targetCurrency);
        }

        return exchangeRate != null ? exchangeRate.getExchangeRate() : null;
    }

    /**
     * 计算交叉汇率
     */
    private BigDecimal calculateCrossRate(String baseCurrency, String targetCurrency, LocalDate effectiveDate) {
        log.debug("尝试计算交叉汇率: {} -> {}", baseCurrency, targetCurrency);

        try {
            // 通过默认基础货币（如USD）计算交叉汇率
            // 获取 baseCurrency -> USD 的汇率
            BigDecimal baseToUsdRate;
            if (baseCurrency.equals(DEFAULT_BASE_CURRENCY)) {
                baseToUsdRate = BigDecimal.ONE;
            } else {
                BigDecimal rate = getDirectExchangeRate(baseCurrency, DEFAULT_BASE_CURRENCY, effectiveDate);
                if (rate != null) {
                    baseToUsdRate = rate;
                } else {
                    // 尝试反向
                    BigDecimal reverseRate = getDirectExchangeRate(DEFAULT_BASE_CURRENCY, baseCurrency, effectiveDate);
                    if (reverseRate != null) {
                        baseToUsdRate = BigDecimal.ONE.divide(reverseRate, 8, RoundingMode.HALF_UP);
                    } else {
                        throw new BusinessException("无法获取 " + baseCurrency + " 到 " + DEFAULT_BASE_CURRENCY + " 的汇率");
                    }
                }
            }

            // 获取 USD -> targetCurrency 的汇率
            BigDecimal usdToTargetRate;
            if (targetCurrency.equals(DEFAULT_BASE_CURRENCY)) {
                usdToTargetRate = BigDecimal.ONE;
            } else {
                BigDecimal rate = getDirectExchangeRate(DEFAULT_BASE_CURRENCY, targetCurrency, effectiveDate);
                if (rate != null) {
                    usdToTargetRate = rate;
                } else {
                    // 尝试反向
                    BigDecimal reverseRate = getDirectExchangeRate(targetCurrency, DEFAULT_BASE_CURRENCY, effectiveDate);
                    if (reverseRate != null) {
                        usdToTargetRate = BigDecimal.ONE.divide(reverseRate, 8, RoundingMode.HALF_UP);
                    } else {
                        throw new BusinessException("无法获取 " + DEFAULT_BASE_CURRENCY + " 到 " + targetCurrency + " 的汇率");
                    }
                }
            }

            // 计算交叉汇率
            BigDecimal crossRate = baseToUsdRate.multiply(usdToTargetRate);
            log.debug("交叉汇率计算结果: {} -> {} = {}", baseCurrency, targetCurrency, crossRate);

            return crossRate;
        } catch (Exception e) {
            log.error("计算交叉汇率失败: {} -> {}", baseCurrency, targetCurrency, e);
            throw new BusinessException("无法获取汇率: " + baseCurrency + " -> " + targetCurrency);
        }
    }

    /**
     * 验证币种是否支持（业务逻辑）
     */
    private boolean isCurrencySupported(String currency) {
        if (StringUtils.isBlank(currency)) {
            return false;
        }

        // 查询该币种是否在汇率表中存在（作为基础币种或目标币种）
        List<ExchangeRate> baseCurrencyRates = iExchangeRateService.getByBaseCurrency(currency);
        List<ExchangeRate> targetCurrencyRates = iExchangeRateService.getByTargetCurrency(currency);

        return !CollectionUtils.isEmpty(baseCurrencyRates) || !CollectionUtils.isEmpty(targetCurrencyRates);
    }

    /**
     * 获取支持的币种列表（业务逻辑）
     */
    private List<String> getSupportedCurrenciesList() {
        // 查询所有有效的汇率记录
        List<ExchangeRate> exchangeRates = iExchangeRateService.getAllActiveRates();

        Set<String> currencies = new HashSet<>();
        for (ExchangeRate rate : exchangeRates) {
            currencies.add(rate.getBaseCurrency());
            currencies.add(rate.getTargetCurrency());
        }

        return new ArrayList<>(currencies);
    }

    /**
     * 获取生效日期
     */
    private LocalDate getEffectiveDate(Date transactionDate, String dateStr) {
        if (transactionDate != null) {
            try {
                String dateString = SIMPLE_DATE_FORMAT.format(transactionDate);
                return LocalDate.parse(dateString, DATE_FORMATTER);
            } catch (Exception e) {
                log.warn("交易日期格式转换失败: {}, 使用当前日期", transactionDate);
            }
        }

        if (StringUtils.isNotBlank(dateStr)) {
            try {
                return LocalDate.parse(dateStr, DATE_FORMATTER);
            } catch (Exception e) {
                log.warn("日期字符串格式错误: {}, 使用当前日期", dateStr);
            }
        }

        // 默认返回null，查询最新汇率
        return null;
    }

    /**
     * 解析日期字符串
     */
    private LocalDate parseDate(String dateStr) {
        if (StringUtils.isBlank(dateStr)) {
            return null;
        }

        try {
            return LocalDate.parse(dateStr, DATE_FORMATTER);
        } catch (DateTimeParseException e) {
            log.warn("日期格式错误: {}, 使用最新汇率", dateStr);
            return null;
        }
    }
}