import {Directive, ElementRef, OnInit, HostListener} from '@angular/core';
import {Router} from '@angular/router';


@Directive({
  selector: '[appDrag]'
})
export class DragDirective implements OnInit {
  isDown = false;
  cur: any = {
    x: 0,
    y: 0
  }; // 记录鼠标点击事件的位置
  disX: number; // 鼠标移动的X轴距离
  disY: number; // 鼠标移动的Y轴距离
  offsetX: number; // 元素的左偏移量
  offsetY: number; // 元素的上偏移量
  pageX: number; // 当前页面的X轴距离
  pageY: number; // 当前页面的Y轴距离
  x: number; // 元素移动之后的X轴位置
  y: number; // 元素移动之后的Y轴位置

  constructor(public el: ElementRef, private router: Router) {
  }


  // 点击事件
  @HostListener('touchstart', ['$event'])
  ontouchstart(event) {
    this.start(event);
  }

  // 监听document移动事件事件
  @HostListener('document:touchmove', ['$event'])
  ontouchmove(event) {
    // 判断该元素是否被点击了。
    this.move(event)
  }

  // 监听document离开事件
  @HostListener('document:touchend', ['$event'])
  ontouchend(event) {
    // 只用当元素移动过了，离开函数体才会触发。
    this.end();
  }

  @HostListener('mouseup', ['$event'])
  onMouseup(event) {
    // 只用当元素移动过了，离开函数体才会触发。
    this.router.navigate(['/']);
  }

  ngOnInit() {
  }

  // 点击开始移动
  start(event) {
    this.isDown = true;
    let touch;
    if (event.touches) {
      touch = event.touches[0];
    } else {
      touch = event;
    }
    this.disX = this.disY = 0;
    this.cur.x = touch.clientX;
    this.cur.y = touch.clientY;
    this.pageX = document.body.clientWidth;
    this.pageY = document.body.clientHeight;
    this.offsetX = this.el.nativeElement.offsetLeft;
    this.offsetY = this.el.nativeElement.offsetTop;
  }

//   移动
  move(event) {
    // 阻止页面的滑动默认事件
    event.preventDefault();
    if (this.isDown) {
      let touch;
      if (event.touches) {
        touch = event.touches[0];
      } else {
        touch = event;
      }
      this.disX = touch.clientX - this.cur.x;
      this.disY = touch.clientY - this.cur.y;
      this.x = this.offsetX + this.disX;
      this.y = this.offsetY + this.disY;
      if (this.x < 0) {
        this.x = 0
      }
      if (this.x > document.body.clientWidth - 60) {
        this.x = document.body.clientWidth - 60;
      }
      if (this.y < 0) {
        this.y = 0
      }
      if (this.y > document.body.clientHeight - 120) {
        this.y = document.body.clientHeight - 120;
      }
      this.el.nativeElement.style.left = this.x + 'px';
      this.el.nativeElement.style.top = this.y + 'px';
    }
  }

//  鼠标释放
  end() {
    this.isDown = false;
    if (this.disX !== 0 || this.disY !== 0) {
      if (parseInt(this.el.nativeElement.style.left, 0) < document.body.clientWidth / 2) {
        this.el.nativeElement.style.left = '0px';
      } else {
        this.el.nativeElement.style.left = 'calc(100% - 70px)';
      }
      return;
    } else {
      // 阻止点透事件
      setTimeout(() => {
        this.router.navigate(['/'])
      }, 300)

    }
  }
}
