package com.yd.api.commission.service.impl;


import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.yd.api.agms.service.AgmsFortuneService;
import com.yd.api.order.vo.SurrenderFortuneRequestVO;
import com.yd.api.commission.service.LifeCommissionService;
import com.yd.api.commission.vo.lifecommission.*;
import com.yd.api.result.CommonResult;
import com.yd.dal.entity.commission.OrderCommissonCheck;
import com.yd.dal.entity.customer.AclCustomerFortune;
import com.yd.dal.entity.order.PoOrder;
import com.yd.dal.mapper.lifecommission.LifeCommissionMapper;
import com.yd.dal.service.customer.AclCustomerFortuneDALService;
import com.yd.dal.service.order.PoOrderDALService;
import com.yd.util.CommonUtil;
import com.yd.util.config.ZHBErrorConfig;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;


@Service
public class LifeCommissionServiceImpl implements LifeCommissionService {

    @Autowired
    private LifeCommissionMapper lifeCommissionMapper;
    @Autowired
    private AclCustomerFortuneDALService customerFortuneDalService;
    @Autowired
    private AgmsFortuneService agmsFortuneService;
    @Autowired
    private PoOrderDALService poOrderDALService;


    @Override
    public ComeCommissionListResponseVO queryComeCommissionList(QueryComeCommissionListRequestVO requestVO) {
        ComeCommissionListResponseVO resp = new ComeCommissionListResponseVO();
        try {
            // 分页结果集
            PageInfo<ComeCommissionVO> pageInfo = this.queryComeCommissionListPage(requestVO);
            resp.setDataList(pageInfo.getList());
            pageInfo.setList(null);
            // 不分页结果集统计总单数，总保费，总来佣
            List<ComeCommissionVO> dataList = lifeCommissionMapper.queryComeCommissionList(requestVO);
            this.getSummaries(resp, dataList);

            resp.setPage(pageInfo);
            resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("800000")));
        } catch (Exception e) {
            e.printStackTrace();
            resp.setCommonResult(new CommonResult(false, e.getMessage()));
        }
        return resp;
    }

    private void getSummaries(ComeCommissionListResponseVO resp, List<ComeCommissionVO> dataList) {
        Integer totalSingular = dataList.size();
        BigDecimal totalOrderPrice = BigDecimal.ZERO;
        BigDecimal totalCommission = BigDecimal.ZERO;

        for (ComeCommissionVO item: dataList) {
            totalOrderPrice = totalOrderPrice.add(new BigDecimal(item.getOrderPrice()));
            totalCommission = totalCommission.add(new BigDecimal(item.getCommissionAmount()));
        }
        resp.setTotalOrderPrice(totalOrderPrice);
        resp.setTotalSingular(totalSingular);
        resp.setTotalCommission(totalCommission);
    }

    @Override
    public CheckComeCommissionResponseVO updateCommissionCheckStatus(CheckComeCommissionRequestVO requestVO) {
        CheckComeCommissionResponseVO resp = new CheckComeCommissionResponseVO();
        List<Long> orderIds = requestVO.getOrderIds();
        String status = requestVO.getCheckStatus();
        String loginId = requestVO.getLoginId();
        String checkBatch = requestVO.getCheckBatch();
        try {
            // 寿险经纪人的财富需要初始化withdraw和pay
            if ("2".equals(status)) {//已比对
                this.checkPass(orderIds, status, loginId, checkBatch);

            } else if ("3".equals(status)) {//已退保
                SurrenderFortuneRequestVO surrenderFortuneRequest = new SurrenderFortuneRequestVO();
                for (Long orderId: orderIds) {
                    surrenderFortuneRequest.setOrderId(orderId);
                    agmsFortuneService.surrenderFortune(surrenderFortuneRequest);
                }
            } else {//待来佣
                this.checkRollBack(orderIds, loginId);
            }
            resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("800000")));

        } catch (Exception e) {
            e.printStackTrace();
            resp.setCommonResult(new CommonResult(false, e.getMessage()));
        }
        return resp;
    }

    private void checkRollBack(List<Long> orderIds, String loginId) throws Exception {
        // 查询对应订单记录
        List<PoOrder> orders = poOrderDALService.findByIds(orderIds);
        // 查询保单下的所有寿险经纪人的fortune记录
        List<AclCustomerFortune> fortuneList = customerFortuneDalService.queryLifeFortuneListByOrderIds(orderIds);

        // 校验数据，若订单下存在已发放的财富记录，则该笔订单来佣比对状态不能改为待来佣
        this.validateRollBackOrders(orders, fortuneList);

        // 设置order记录为待来佣
        for (PoOrder o:orders) {
            o.setCommissionCheckId(null);
            o.setCommissionCheckStatus("1");
            o.setCommissionCheckAt(new Date());
            o.setCommissionCheckBy(Long.getLong(loginId));
            poOrderDALService.update(o);
        }
        // 批量设置fortune为不可发佣
        fortuneList.forEach(f -> {
            f.setCommissionPayoutStatus("1");
            f.setCommissionPayoutAt(new Date());
            f.setCommissionPayoutBy(Long.getLong(loginId));
        });
        if (fortuneList.size() > 0) {
            customerFortuneDalService.updateBatch(fortuneList);
        }
    }

    private void validateRollBackOrders(List<PoOrder> orders, List<AclCustomerFortune> fortuneList) throws Exception {
        StringBuilder stringBuilder = new StringBuilder();
        Map<Long, String> orderIdPolicyNoMap = new HashMap<>();
        for (PoOrder o:orders) {
            orderIdPolicyNoMap.put(o.getId(), o.getPolicyNo());
            if ("1".equals(o.getCommissionCheckStatus()) || "3".equals(o.getCommissionCheckStatus())) {
                stringBuilder.append(o.getPolicyNo()+" ");
            }
        }
        if (StringUtils.isNotEmpty(stringBuilder.toString())) {
            stringBuilder.append("待来佣和已退保的订单不能改为待来佣");
            throw new Exception(stringBuilder.toString());
        }
        for (AclCustomerFortune f:fortuneList) {
            if ("4".equals(f.getCommissionPayoutStatus())) {
                stringBuilder.append(orderIdPolicyNoMap.get(f.getOrderId())+" ");
            }
        }
        if (StringUtils.isNotEmpty(stringBuilder.toString())) {
            stringBuilder.append("订单下存在已发放的财富记录，不能改为待来佣");
            throw new Exception(stringBuilder.toString());
        }
    }

    private void checkPass(List<Long> orderIds, String status, String loginId, String checkBatch) {
        // 插入批次表信息
        Long checkBatchId = this.getOrderCommissionCheckBatch(checkBatch, loginId);
        // 设置order记录的CommissionCheckId
        this.setOrderCommissionCheckId(orderIds, status, checkBatchId, loginId);
        // 查询保单下的所有寿险经纪人的fortune记录
        List<AclCustomerFortune> fortuneList = customerFortuneDalService.queryLifeFortuneListByOrderIds(orderIds);
        if (fortuneList.size() == 0) {
            return;
        }
        // 批量设置fortune为可发佣
        fortuneList.forEach(f -> {
            f.setCommissionPayoutStatus("2");
            f.setCommissionPayoutAt(new Date());
            f.setCommissionPayoutBy(Long.valueOf(loginId));
        });
        customerFortuneDalService.updateBatch(fortuneList);
        // 初始化withdraw和pay
        agmsFortuneService.canPaymentUpDate("2", Long.valueOf("-1"), Long.valueOf(loginId), fortuneList);
    }

    private PageInfo<ComeCommissionVO> queryComeCommissionListPage(QueryComeCommissionListRequestVO requestVO) {
        PageInfo<ComeCommissionVO> page = requestVO.getPage();

        String orderBy = "orderDate asc";

        PageHelper.startPage(page.getPageNum(), page.getPageSize(), orderBy);

        List<ComeCommissionVO> dataList = lifeCommissionMapper.queryComeCommissionList(requestVO);

        PageInfo<ComeCommissionVO> pageInfo = new PageInfo<>(dataList);

        return pageInfo;
    }

    private void setOrderCommissionCheckId(List<Long> orderIds, String status, Long checkBatchId, String loginId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("checkId", checkBatchId);
        paramMap.put("status", status);
        // 获取当前时间
        Date curDate = new Date();
        paramMap.put("checkAt", curDate);
        paramMap.put("checkBy", loginId);

        lifeCommissionMapper.setOrderCommissionCheckId(orderIds, paramMap);
    }

    private Long getOrderCommissionCheckBatch(String OrderCommissionCheckBatch, String loginId) {
        OrderCommissonCheck orderCommissonCheck = lifeCommissionMapper.findByCheckYearmonth(OrderCommissionCheckBatch);
        if (orderCommissonCheck == null){
            // 获取当前时间
            Date curDate = new Date();
            // 获取代表批次的年月
            String yearmonth = CommonUtil.dateParseString(curDate, "yyyy-MM");

            orderCommissonCheck = new OrderCommissonCheck();
            orderCommissonCheck.setCheckYearmonth(yearmonth);
            orderCommissonCheck.setCreatedAt(curDate);
            orderCommissonCheck.setCreatedBy(loginId);
            orderCommissonCheck.setUpdatedAt(curDate);
            orderCommissonCheck.setUpdatedBy(loginId);
            lifeCommissionMapper.insertOrderCommissionCheck(orderCommissonCheck);
        }
        return orderCommissonCheck.getId();
    }
}
